
/* communication routines, all sorts */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <termios.h>
#include <stdlib.h>

#include "prot.h"
#include "comm.h"
#include "display.h"
#include "vars.h"

static int calc_fd;
static void calc_xmit_pack(TI_Packet *pack);
static void calc_recv_pack(TI_Packet *pack);
static int ti_isok(void);
static inline int ti_read(void *buf, size_t size);
static inline int ti_write(void *buf, size_t size);

/*
#define calc_recv_pack(x) ti_read(x, sizeof(TI_Packet));
#define calc_xmit_pack(x) ti_write(x, sizeof(TI_Packet));
*/

/* define to get incoming hex dumps */
#undef DEBUG_READ
#undef DEBUG_WRITE 

/* makes packets simpler */
#define mkpkt(a, y, z) { \
     a.device = (pkt_id & 0x0f); \
     a.command = y; \
     a.length = z; \
}

/* global directory structure */
extern struct ti_dirent *dir;

int calc_model;
int pkt_id;

int init_comm(int calc, char *device) {

     TI_Packet packet;
     int status;
     struct termios termset;

     if((calc != 92) && (calc != 89)) {
	  pmain("Calculator must be 92 or 89!\n");
	  endwin();
	  exit(1);
     }

     calc_model = calc;

     if(calc == 92)
	  pkt_id = 0x89;
     else
	  pkt_id = 0x98;

     if((calc_fd = open(device, O_RDWR)) == -1) {
	  pmain("Failed to open %s\n", device);
	  endwin();
	  exit(1);
     }

     /* if we opened a real serial port, set the terminal modes and
	use that */
     if(!tcgetattr(calc_fd, &termset)) {
	  cfmakeraw(&termset);
	  cfsetispeed(&termset, B9600);
	  cfsetospeed(&termset, B9600);
	  tcsetattr(calc_fd, TCSANOW, &termset);
     }

     mkpkt(packet, COM_TEST92, 0);

     status = ti_write(&packet, sizeof(TI_Packet));
     status = ti_read(&packet, sizeof(TI_Packet));

     if(packet.command != COM_OK) {
	  endwin();
	  printf("Unable to contact calculator\n");
	  exit(1);
     }

     pmain("Communication channel initialized\n");
     
     return 0;

}

/* sends a keystroke to the 89/92 (89 hardcoded now) in "command mode" */

int command_key(unsigned short key) {

     TI_Packet packet;

     mkpkt(packet, COM_DIRECT, key);

     calc_xmit_pack(&packet);
     calc_recv_pack(&packet);

     return 0;
}

static void calc_xmit_pack(TI_Packet *pack) {

     if(ti_write(pack, sizeof(TI_Packet)) < sizeof(TI_Packet)) {
	  endwin();
	  printf("Packet could not be transmitted, driver or device fault\n");
	  exit(1);
     }

     return;
}

static void calc_recv_pack(TI_Packet *pack) {

     int c;

     if((c = ti_read(pack, sizeof(TI_Packet))) < sizeof(TI_Packet)) {
	  endwin();
	  printf("Incomplete packet recieved from calculator, terminating\n");
	  printf("(%d bytes, expected %d)\n", c, sizeof(TI_Packet));
	  exit(1);
     }

}

void dir_full(void) {

     TI_Packet packet;
     TI_Varhead92 folder;
     unsigned long chksum;
     unsigned char name[8];
     struct ti_dirent *ret;
     struct ti_dirent *cur;

     TI_Dir92 entry;
     int x;

     ti_isok();

     mkpkt(packet, COM_REQ, REQ_DIR);
     calc_xmit_pack(&packet);
     ti_write(req_fulldir, 8);
	
     /* this packet should say OK */
     calc_recv_pack(&packet);
     
     /* now we're sent the current folder */
     calc_recv_pack(&packet);
     
     ti_read(&folder, sizeof(TI_Varhead92));
     ti_read(name, folder.namelen);
     name[folder.namelen] = 0;
     ti_read(&chksum, 4);

     pmain("Current folder: %s\n", name);

     /* get the entries... */
     do {
	  mkpkt(packet, COM_OK, 0);
	  calc_xmit_pack(&packet);
	  mkpkt(packet, COM_WAIT, 0xffff); /* example shows
						      length set to
						      0xffff ? */
	  calc_xmit_pack(&packet);
	  calc_recv_pack(&packet);
	  calc_recv_pack(&packet);
	  ti_read(&entry, sizeof(TI_Dir92));
	  if(entry.type == VAR92_GROUP)
	       pmain("%s\n", entry.name);
	  else
	       pmain("\t%-16s%d\t%c\t%d\n", entry.name, do_size(entry.size), 
		     filetypes_92[entry.type], entry.status);
	  mkpkt(packet, COM_OK, 0);
	  calc_xmit_pack(&packet);
	  calc_recv_pack(&packet);

     } while(packet.command != COM_EOT);
     
     /* end with an ok */
     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);

     return;
}

void dir_folders(void) {

     TI_Packet packet;
     TI_Varhead92 folder;
     unsigned long chksum;
     unsigned char name[8];
     unsigned char data[15];

     TI_Dir92 entry;
     int x;

     ti_isok();
     
     mkpkt(packet, COM_REQ, REQ_DIR);
     calc_xmit_pack(&packet);
     ti_write(req_folders, 8);

     /* this packet should say OK */
     calc_recv_pack(&packet);
     
     /* now we're sent the current folder */
     calc_recv_pack(&packet);
     
     ti_read(&folder, sizeof(TI_Varhead92));
     ti_read(name, folder.namelen);
     name[folder.namelen] = 0;
     ti_read(&chksum, 4);

     pmain("Current folder: %s\n", name);

     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
    
     /* hex dump of ti's transmission shows 0xffff */
     mkpkt(packet, COM_WAIT, 0xffff); 
     calc_xmit_pack(&packet);
     calc_recv_pack(&packet);
     calc_recv_pack(&packet);

     /* ti sends 4 bytes of nulls before the data */
     x = ti_read(data, 4);
     while(x < packet.length) {
	  x += ti_read(data, 14);
	  pmain("%-10s\n", data);
     }
     /* now the ti sends a checksum */
     ti_read(&chksum, 2);
     
     /* end with an ok */
     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     
     /* now the TI will send an EOT */
     calc_recv_pack(&packet);
     
     return;
}

int ti_isok(void) {

     TI_Packet pack;

     mkpkt(pack, COM_TEST92, 0);

     calc_xmit_pack(&pack);
     calc_recv_pack(&pack);
     
     return pack.length;
}
     

static inline int ti_read(void *buf, size_t size) {

     int r;
     int j = 0;
#ifdef DEBUG_READ
     unsigned char *bleh = (unsigned char *)buf;
     int i = 0;
#endif

     while((r = read(calc_fd, buf, size)) == 0) {
	  if(j > 5)
	       return 0;
	  j++;
     }


#ifdef DEBUG_READ

     while(i < r) {
	  pmain("%x ", bleh[i]);
	  i++;
     }
     pmain("(of %d)\n", size);
#endif

     return r;

}

static inline int ti_write(void *buf, size_t size) {

     int ret;
     
#ifdef DEBUG_WRITE
     int i = 0;
     unsigned char *bleh = (unsigned char *)buf;

     while(i < size) {
	  pmain("%x ", bleh[i]);
	  i++;
     }
     pmain("(of %d)\n", size);
#endif
     
     if(size < 2048)
	  return write(calc_fd, buf, size);
     else { 
	  void *pos = buf;
	  int rem = size;
	  int sent;

	  while(rem > 0) {
	       sent = write(calc_fd, pos, rem);
	       pmain("wrote %d bytes\n", rem);
	       rem -= sent;
	       pos += sent;
	  }

	  return sent;
     }
}

/* this would do a go into recieve mode, for the ti op to hit send on
   some variable */

int link_recv(void) {

     TI_Packet packet;
     TI_Varhead92 head;
     TI_Varjunk92 junk; 
     char vname[255];
     void *data;
     time_t fail;
     unsigned int chksum;

     pmain("Waiting for transmission...\n");

     /* loop for a bit until the ti89 says something */
    
     fail = time(NULL)+15; /* 15 seconds */
     
     while(ti_read(&packet, sizeof(TI_Packet)) == 0) {
	  if(time(NULL) > fail)
	       return -1;
	  /* sleep for 1/10 second */
	  usleep(100000);
     }

     /* was this a var packet */
     if(packet.command != COM_VAR) {
	  return -1;
     }
     
     ti_read(&head, sizeof(TI_Varhead92));
     ti_read(vname, head.namelen);
     ti_read(&chksum, 4);
     vname[head.namelen] = 0;
     pmain("Receiving %s.%d%c -- ", vname, calc_model,
	   filetypes_92[head.type]);

     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     mkpkt(packet, COM_WAIT, 0);
     calc_xmit_pack(&packet);
     
     calc_recv_pack(&packet);
     calc_recv_pack(&packet);
     if(packet.command != COM_DATA)
	  return -1;

     ti_read(&junk, sizeof(TI_Varjunk92));
     data = malloc(rev_size(junk.size));
     pmain("%d bytes\n", rev_size(junk.size));
     ti_read(data, rev_size(junk.size));

     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     calc_recv_pack(&packet);
     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);

     free(data);

     return 0;

}

/* gets a vairable by name */
int get_var(char *name, int type) {

     TI_Packet packet;
     TI_Varhead92 var;
     TI_Varjunk92 bleh;
     char *dump[20];

     unsigned int chksum;
     void *data;
     int size;
     int x;

     ti_isok();

     mkpkt(packet, COM_REQ, strlen(name)+sizeof(TI_Varhead92));
     calc_xmit_pack(&packet);

     /* clear the varhead */
     memset(&var, 0, sizeof(TI_Varhead92));
     
     var.type = type;
     var.namelen = strlen(name);

     ti_write(&var, sizeof(TI_Varhead92));
     ti_write(name, strlen(name));

     chksum = calc_sum(&var, sizeof(TI_Varhead92)) + calc_sum(name,
							      strlen(name));
     ti_write(&chksum, 2);

     /* we ought to get an ok, wonder what it says on an invalid var */
     calc_recv_pack(&packet);
     calc_recv_pack(&packet);
     
     /* it sends a varhead */
     ti_read(&var, sizeof(TI_Varhead92));
     ti_read(dump, var.namelen);
     ti_read(&chksum, 3); /* 3 here.  no idea why */
     
     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     mkpkt(packet, COM_WAIT, 0xffff);
     calc_xmit_pack(&packet);

     /* ok again */
     calc_recv_pack(&packet);
     /* and now data */
     calc_recv_pack(&packet);
     
     ti_read(&bleh, sizeof(TI_Varjunk92));
     size = rev_size(bleh.size);

     data = alloca(size);
     ti_read(data, size);
     ti_read(&chksum, 2);

     /* and now we have it */

     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     /* should say eot */
     calc_recv_pack(&packet);
     
     
     return save_var(name, type, data, size);
     
        
}

int send_var(char *filename) {

     TI_Packet packet;
     TI_Varhead92 head;
     TI_Varjunk92 junk;

     void *data;
     int len;
     char varname[8];
     char folder[8];
     char fullname[20];
     unsigned short chksum;
     
     int i;

     if((data = load_var(filename, &len, varname, folder)) == NULL)
	  return -1;

     /* clear the header */
     memset(&head, 0, sizeof(TI_Varhead92));

     head.namelen = sprintf(fullname, "%s\\%s", folder, varname); 
     head.length = len + 2; /* why ? */
     head.type = 0xff;
     
     for(i = 0; i < strlen(filetypes_92); i++)
	  if(filetypes_92[i] == filename[strlen(filename)-1])
	       head.type = i;

     if(i == 0xff) {
	  pmain("Invalid variable type \"%c\"",
		filename[strlen(filename)-1]);
	  return -1;
     }

     mkpkt(packet, COM_SEND, head.namelen+sizeof(TI_Varhead92)+1);
     calc_xmit_pack(&packet);
     ti_write(&head, sizeof(TI_Varhead92));
     ti_write(fullname, head.namelen+1);
     chksum = calc_sum(&head, sizeof(TI_Varhead92)) +
	  calc_sum(fullname, head.namelen);
     ti_write(&chksum, 2);

     /* ti should send an OK then a WAIT */
     calc_recv_pack(&packet);
     calc_recv_pack(&packet);

     /* we send OK then the data */
     mkpkt(packet, COM_OK, 0);
     calc_xmit_pack(&packet);
     mkpkt(packet, COM_DATA, len+sizeof(TI_Varjunk92));
     calc_xmit_pack(&packet);

     memset(&junk, 0, sizeof(TI_Varjunk92));
     junk.size[0] = (len & 0xff00) >> 8;
     junk.size[1] = len & 0x00ff;

     ti_write(&junk, sizeof(TI_Varjunk92));
     ti_write(data, len);
     chksum = calc_sum(&junk, sizeof(TI_Varjunk92)) + calc_sum(data,
							       len);
     ti_write(&chksum, 2);

     /* ti should say ok */
     calc_recv_pack(&packet);
     mkpkt(packet, COM_EOT, 0);
     calc_xmit_pack(&packet);
     calc_recv_pack(&packet);

     free(data);
     return 0;
     
}

